package cmd

import (
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"sort"
	"strings"
)

// GetAllFilePathByExts 从指定路径获取指定后后缀的文件列表,ext:.go
func GetAllFilePathByExts(currentPath string, exts []string) (filteredPaths []string, err error) {

	Paths, err := GetAllFilePath([]string{}, currentPath)
	if err != nil {
		XWarning(fmt.Sprintf("GetAllFilePathByExt error : %v\n", err))
		return
	}

	// 过滤是否包含指定后缀
	for _, path := range Paths {
		for _, ext := range exts {
			if strings.HasSuffix(path, ext) {
				filteredPaths = append(filteredPaths, path)
			}
		}

	}

	return
}

// GetAllFilePathByExt 从指定路径获取指定后后缀的文件列表,ext:.go
func GetAllFilePathByExt(currentPath string, ext string) (filteredPaths []string, err error) {

	Paths, err := GetAllFilePath([]string{}, currentPath)
	if err != nil {
		XWarning(fmt.Sprintf("GetAllFilePathByExt error : %v\n", err))
		return
	}

	// 过滤是否包含指定后缀
	for _, path := range Paths {

		if strings.HasSuffix(path, ext) {
			filteredPaths = append(filteredPaths, path)
		}

	}

	return
}

// GetAllFilePath 获取指定目录内的文件和子目录的文件,递归获取
func GetAllFilePath(filePaths []string, currentPath string) ([]string, error) {
	//2021-12-23
	files, err := ioutil.ReadDir(currentPath)

	if err != nil {
		log.Printf("filepath.Glob error : %v\n", err)
		return filePaths, err
	}
	for _, f := range files {
		if f.IsDir() {
			ret, err := GetAllFilePath(filePaths, filepath.Join(currentPath, f.Name()))
			if err != nil {
				log.Printf("GetAllFilePath eror: %v\n", err)
				return filePaths, err
			} else {
				filePaths = append(filePaths, ret...)
			}

		} else {
			filePaths = append(filePaths, filepath.Join(currentPath, f.Name()))
		}

	}
	return filePaths, nil
}

// FileExists 检测文件是否存在
func FileExists(filePath string) (bool, error) {
	_, err := os.Stat(filePath)
	if err == nil {
		return true, nil
	}
	if os.IsNotExist(err) {
		return false, nil
	}
	return false, err
}

// GetFileFromFolder 获取子文件夹里的文件名列表
func GetFileFromFolder(folder string) ([]string, error) {
	filePaths := []string{}
	files, err := ioutil.ReadDir(folder)

	if err != nil {
		log.Printf("filepath.Glob error : %v\n", err)
		return filePaths, err
	}
	for _, f := range files {
		filePaths = append(filePaths, filepath.Join(folder, f.Name()))
	}
	return filePaths, nil
}

// RemoveFiles 删除指定路径和指定后缀的文件
func RemoveFiles(folder, suffix string) error {
	filePaths, err := GetFileFromFolder(folder)

	if err != nil {
		log.Printf("GetFileFromFolder error : %v\n", err)
		return err
	}

	for _, filePath := range filePaths {

		if filePath[len(filePath)-len(suffix):] == suffix {
			err = os.Remove(filePath)
			if err != nil {
				log.Printf("Remove error : %v\n", err)
			}

		}
	}

	return nil
}

// GetSubTitleByModTime 从指定路径获取字幕文件,按mod时间排序
func GetSubTitleByModTime(folder string) (string, error) {
	//2021-11-03
	filePaths := []string{}
	files, err := ioutil.ReadDir(folder)

	if err != nil {
		log.Printf("filepath.Glob error : %v\n", err)
		return filePaths[0], err
	}

	//按mod时间排序
	sort.Slice(files, func(i, j int) bool {
		flag := false
		if files[i].ModTime().After(files[j].ModTime()) {
			flag = true
		} else if files[i].ModTime().Equal(files[j].ModTime()) {
			if files[i].Name() < files[j].Name() {
				flag = true
			}
		}
		return flag
	})

	for _, f := range files {
		filePaths = append(filePaths, filepath.Join(folder, f.Name()))
	}

	return filePaths[0], err
}
